<?php
date_default_timezone_set('America/Bogota');

// URLs fijas
const TCC_WSDL_URL      = 'http://clientes.tcc.com.co/servicios/informacionremesas.asmx?wsdl';
const MONITOR_ISDIN_URL = 'http://129.146.151.238/monitor_isdin/monitor_isdin.php';
const MONITOR_ISDIN_API_REINTEGRAR = 'http://129.146.161.23/BackEnd_Orion/reintegrarIsdinWebeat.php';

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
if ($method !== 'GET') {
    header('Content-Type: application/json; charset=utf-8');
    header('Allow: GET');
    http_response_code(405);
    echo json_encode([
        'status' => false,
        'message' => 'Method Not Allowed. Use GET method.',
    ]);
    exit;
}

/**
 * Consulta el tracking en TCC y retorna:
 * - true  => si tiene novedades
 * - false => si NO tiene novedades o si algo falla
 */
function consultarTrackingEstadoTieneNovedades(string $trackingNumber): bool
{
    try {
        $client = new \SoapClient(TCC_WSDL_URL);

        $result = $client->ConsultarInformacionRemesasEstados([
            "Clave"    => "BOGPHAREXSA0502",
            "remesas"  => ["Remesa" => ["numeroremesa" => $trackingNumber]],
            "Respuesta" => "0",
            "Mensaje"  => "0"
        ]);

        // Si el servicio responde error, asumimos que NO hay novedades (para no romper flujo)
        if (!isset($result->Respuesta) || (int)$result->Respuesta !== 0) {
            return false;
        }

        if (empty((array) $result->remesasrespuesta)) {
            return false;
        }

        $remesaEstados = $result->remesasrespuesta->RemesaEstados ?? null;
        if (!$remesaEstados) {
            return false;
        }

        // Novedades: remesasrespuesta->RemesaEstados->listanovedades->Novedad
        if (!isset($remesaEstados->listanovedades->Novedad)) {
            return false;
        }

        $novedadesRaw = $remesaEstados->listanovedades->Novedad;

        // Normalizar a array
        if (!is_array($novedadesRaw)) {
            $novedadesRaw = [$novedadesRaw];
        }

        // Si hay al menos una novedad ejecutada, consideramos que hay incidencia
        foreach ($novedadesRaw as $novedad) {
            $n = (array) $novedad;

            $estadoNovedad = strtoupper(trim($n['estadonovedad'] ?? ''));
            // Puedes ajustar esta condición según negocio
            if ($estadoNovedad === 'EJECUTADA' || $estadoNovedad === 'EJECUTADO' || $estadoNovedad === '') {
                return true;
            }
        }

        return true; // hay novedades, aunque no se haya revisado el estado puntualmente
    } catch (\Throwable $th) {
        // Lo tratamos como "sin info de novedades" -> false
        return false;
    }
}

try {
    header('Content-Type: application/json; charset=utf-8');

    if ($_GET['action'] === 'fetch') {
        $response = json_decode(file_get_contents(MONITOR_ISDIN_URL), true);

        if (!$response || !isset($response['status']) || !$response['status']) {
            echo json_encode([
                'status'  => false,
                'message' => 'Failed to fetch data from external service.'
            ]);
            exit;
        }

        $startDate = $_GET['startDate'] ?? null;
        $endDate   = $_GET['endDate'] ?? null;
        $estado    = $_GET['estado'] ?? null;

        // 1) Filtrar SOLO por fechas
        $filteredData = array_filter($response['data'], function ($item) use ($startDate, $endDate) {
            $fechaRuta = date('Y-m-d', strtotime($item['FECHA_RUTA']));

            if ($startDate && $fechaRuta < $startDate) return false;
            if ($endDate && $fechaRuta > $endDate) return false;

            return true;
        });


        // 2) Procesar cada registro: CEDI, fecha y ESTADO_INTEGRACION (E / T / I)
        $processedData = array_map(function ($item) {

            $CEDISufix = substr($item['CEDI'], 5);
            $CEDI = match ($CEDISufix) {
                'P001' => 'CELTA',
                'P002' => 'INTEXONA',
                'P003' => 'ALSACIA',
                'P004' => 'CARTAGENA',
                default => 'N/A'
            };
            $item['CEDI'] = $CEDI;

            $item['FECHA_RUTA'] = date('Y-m-d H:i:s', strtotime($item['FECHA_RUTA']));

            if ($item['FECHA_ENTREGA'] !== 'No entregado') {
                // Pedido entregado
                $estadoIntegracion = 'E';
            } else {
                // En tránsito
                $esTcc = (strpos($item['TRANSPORTADORA'], 'TCC') !== false);

                if ($esTcc) {
                    $tieneNovedades = consultarTrackingEstadoTieneNovedades($item['REMESA']);

                    if ($tieneNovedades) {
                        $estadoIntegracion = 'I'; // Incidencia
                    } else {
                        $estadoIntegracion = 'T'; // Transporte normal
                    }
                } else {
                    $estadoIntegracion = 'T';
                }
            }

            $item['ESTADO_INTEGRACION'] = $estadoIntegracion;

            return $item;
        }, $filteredData);

        // 3) Filtro por estado E / T / I (opcional)
        if ($estado) {
            $estadoFiltro = strtoupper($estado);

            $processedData = array_filter($processedData, function ($item) use ($estadoFiltro) {
                $code = strtoupper($item['ESTADO_INTEGRACION'] ?? '');

                if (in_array($estadoFiltro, ['E', 'T', 'I'], true)) {
                    return $code === $estadoFiltro;
                }

                return match ($estadoFiltro) {
                    'ENTREGADO'   => $code === 'E',
                    'TRANSITO'  => $code === 'T',
                    'INCIDENCIA'  => $code === 'I',
                    default       => true,
                };
            });
        }


        echo json_encode([
            'status' => true,
            'data'   => array_values($processedData)
        ]);
    }

    if ($_GET['action'] === 'reintegrar') {
        $numeroPedido = $_GET['numeroPedido'] ?? null;

        if (!$numeroPedido) {
            echo json_encode([
                'status'  => false,
                'message' => 'Número de pedido no proporcionado.'
            ]);
            exit;
        }

        $response = json_decode(file_get_contents(MONITOR_ISDIN_API_REINTEGRAR . "?pedido=" . urlencode($numeroPedido)), true);

        if (!$response || !isset($response['state']) || $response['state'] !== 'success') {
            echo json_encode([
                'status'  => false,
                'message' => 'Error al reintegrar el pedido.'
            ]);
            exit;
        }

        echo json_encode([
            'status'  => true,
            'message' => "Pedido {$numeroPedido} reintegrado exitosamente.",
            'data'    => $response['data']
        ]);
    }

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status'   => false,
        'message' => $e->getMessage()
    ]);
    exit;
}
