$(document).ready(function () {

  // 🔹 Param type desde la URL
  const urlParams = new URLSearchParams(window.location.search);
  const typeParam = urlParams.get("type");

  const urlLoadInfo = typeParam
    ? "../../back-end/monitor_isdin/monitor_isdin.php"
    : "back-end/monitor_isdin/monitor_isdin.php";

  // 🔹 Inicializar DataTable
  const datatable = $("#list_pedidos").DataTable({
    columns: [
      { data: "NUMEROPEDIDO", title: "Número Pedido" },
      { data: "TRANSPORTADORA", title: "Transportadora" },
      { data: "REMESA", title: "Remesa" },
      { data: "FECHA_RUTA", title: "Fecha Ruta" },
      { data: "FECHA_ENTREGA", title: "Fecha Entrega" },
      { data: "FECHA_EVENTO", title: "Fecha Evento" },
      { data: "DIRECCION", title: "Dirección" },
      { data: "CEDI", title: "CEDI" },
      { data: "ESTADO_INTEGRACION", title: "Estado I" },
      { data: "ACCIONES", title: "Acciones" },
    ],
    paging: true,
    searching: true,
    ordering: false,
    info: true,
    responsive: true,
    pageLength: 10,
    lengthMenu: [10, 25, 50, 100],
    language: {
      url: "//cdn.datatables.net/plug-ins/1.10.21/i18n/Spanish.json",
    },
    dom: "Bfrtip",
    buttons: [
      {
        extend: 'csv',
        text: 'Exportar CSV',
        className: 'btn btn-success',
        exportOptions: {
          columns: ':visible'
        }
      },
    ]
  });

  // 🔹 Cargar información (lee filtros directamente del DOM)
  async function loadInfo() {
    try {
      const startDate = $('#startDate').val();
      const endDate = $('#endDate').val();
      const estado = $('#estadoFilter').val();

      const params = new URLSearchParams();
      if (startDate) params.append('startDate', startDate);
      if (endDate) params.append('endDate', endDate);
      if (estado) params.append('estado', estado);

      const response = await fetch(`${urlLoadInfo}?action=fetch&${params.toString()}`);
      const { status, data } = await response.json();

      if (status) {
        // Actualizar tabla
        datatable.clear().rows.add(
          data.map(item => ({
            ...item,
            ACCIONES: `<button class="btn btn-primary revisar-novedades" data-numeropedido="${item.NUMEROPEDIDO}">Reintegrar</button>`
          }))
        ).draw();

        // Totales por estado de integración
        const countE = data.filter(item => item.ESTADO_INTEGRACION === 'E').length;
        const countT = data.filter(item =>
          item.ESTADO_INTEGRACION === 'T' || item.ESTADO_INTEGRACION === 'I'
        ).length;

        $('#countE').text(countE);
        $('#countT').text(countT);
        $('#countTotal').text(data.length);
      } else {
        console.error('Error en la respuesta del servidor (status=false)');
      }
    } catch (error) {
      console.error('Error al cargar información:', error);
    }
  }

  // 🔹 Reintegrar pedido
  async function reintegrar(numeroPedido) {
    try {
      const response = await fetch(
        `${urlLoadInfo}?action=reintegrar&numeroPedido=${encodeURIComponent(numeroPedido)}`
      );
      const result = await response.json();

      if (result.status) {
        Swal.fire({
          icon: 'success',
          title: 'Éxito',
          text: result.message || 'Pedido reintegrado exitosamente.',
        });
        loadInfo(); // Recargar información
      } else {
        Swal.fire({
          icon: 'warning',
          title: 'Atención',
          text: 'Ocurrió un error al reintegrar el pedido',
        });
      }
    } catch (error) {
      console.error('Error al reintegrar el pedido:', error);
      Swal.fire({
        icon: 'error',
        title: 'Error',
        text: 'Ocurrió un error al reintegrar el pedido.',
      });
    }
  }

  // 🔹 Click en botón "Filtrar"
  $("#filterBtn").on("click", function () {
    loadInfo();
  });

  // 🔹 Click en botón "Resetear filtros"
  $("#resetBtn").on("click", function () {
    $("#startDate").val("");
    $("#endDate").val("");
    $("#estadoFilter").val("");
    loadInfo();
  });

  // 🔹 Delegación de evento para el botón "Reintegrar"
  $('#list_pedidos').on('click', '.revisar-novedades', function () {
    const numeroPedido = $(this).data('numeropedido');
    if (!numeroPedido) {
      console.error('No se encontró data-numeroPedido en el botón');
      return;
    }
    reintegrar(numeroPedido);
  });

  // 🔹 Carga inicial
  loadInfo();
});
